<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>CIP Token Swap</title>
    <style>
        body {
            font-family: 'Arial', sans-serif;
            max-width: 500px;
            margin: 0 auto;
            padding: 20px;
            background-color: #f5f5f5;
            color: #333;
        }
        
        .swap-container {
            background-color: white;
            border-radius: 12px;
            padding: 20px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
        }
        
        h1 {
            text-align: center;
            color: #3a3a3a;
            margin-bottom: 25px;
        }
        
        .token-input {
            margin-bottom: 20px;
        }
        
        .input-header {
            display: flex;
            justify-content: space-between;
            margin-bottom: 8px;
        }
        
        .balance {
            font-size: 14px;
            color: #666;
        }
        
        .input-container {
            display: flex;
            border: 1px solid #ddd;
            border-radius: 8px;
            overflow: hidden;
        }
        
        input {
            flex: 1;
            border: none;
            padding: 12px;
            font-size: 18px;
            outline: none;
        }
        
        .token-selector {
            display: flex;
            align-items: center;
            padding: 0 12px;
            background-color: #f0f0f0;
            cursor: pointer;
        }
        
        .token-selector span {
            margin-left: 8px;
            font-weight: bold;
        }
        
        button {
            width: 100%;
            padding: 14px;
            border: none;
            border-radius: 8px;
            font-size: 16px;
            font-weight: bold;
            cursor: pointer;
            transition: background-color 0.3s;
        }
        
        .connect-btn {
            background-color: #4CAF50;
            color: white;
        }
        
        .connect-btn:hover {
            background-color: #45a049;
        }
        
        .swap-btn {
            background-color: #2196F3;
            color: white;
            margin-top: 10px;
            display: none;
        }
        
        .swap-btn:hover {
            background-color: #0b7dda;
        }
        
        .approve-btn {
            background-color: #ff9800;
            color: white;
            margin-top: 10px;
            display: none;
        }
        
        .approve-btn:hover {
            background-color: #e68a00;
        }
        
        .wallet-options {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-color: rgba(0, 0, 0, 0.5);
            z-index: 100;
            justify-content: center;
            align-items: center;
        }
        
        .wallet-modal {
            background-color: white;
            padding: 20px;
            border-radius: 12px;
            width: 300px;
        }
        
        .wallet-option {
            display: flex;
            align-items: center;
            padding: 12px;
            margin: 8px 0;
            border: 1px solid #ddd;
            border-radius: 8px;
            cursor: pointer;
        }
        
        .wallet-option:hover {
            background-color: #f5f5f5;
        }
        
        .wallet-option img {
            width: 24px;
            height: 24px;
            margin-right: 12px;
        }
        
        .close-btn {
            text-align: right;
            margin-bottom: 10px;
            cursor: pointer;
            font-weight: bold;
        }
        
        .arrow-down {
            text-align: center;
            margin: 10px 0;
            font-size: 20px;
        }
        
        .status-message {
            margin-top: 15px;
            padding: 10px;
            border-radius: 8px;
            display: none;
        }
        
        .success {
            background-color: #dff0d8;
            color: #3c763d;
        }
        
        .error {
            background-color: #f2dede;
            color: #a94442;
        }
        
        .loading {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 3px solid rgba(0,0,0,.3);
            border-radius: 50%;
            border-top-color: #000;
            animation: spin 1s ease-in-out infinite;
            margin-right: 10px;
        }
        
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
    </style>
</head>
<body>
    <div class="swap-container">
        <h1>CIP Token Swap</h1>
        
        <div class="token-input">
            <div class="input-header">
                <span>You pay</span>
                <span class="balance" id="oldTokenBalance">Balance: 0</span>
            </div>
            <div class="input-container">
                <input type="number" id="oldTokenAmount" placeholder="0.0">
                <div class="token-selector" id="oldTokenSelector">
                    <span>OLD</span>
                </div>
            </div>
        </div>
        
        <div class="arrow-down">↓</div>
        
        <div class="token-input">
            <div class="input-header">
                <span>You receive</span>
                <span class="balance">NEW Token</span>
            </div>
            <div class="input-container">
                <input type="number" id="newTokenAmount" placeholder="0.0" readonly>
                <div class="token-selector" id="newTokenSelector">
                    <span>NEW</span>
                </div>
            </div>
        </div>
        
        <button class="connect-btn" id="connectWalletBtn">Connect Wallet</button>
        <button class="approve-btn" id="approveBtn">Approve OLD Tokens</button>
        <button class="swap-btn" id="swapBtn">Swap</button>
        
        <div class="status-message" id="statusMessage"></div>
    </div>
    
    <div class="wallet-options" id="walletOptions">
        <div class="wallet-modal">
            <div class="close-btn" id="closeWalletOptions">✕</div>
            <h3>Connect Wallet</h3>
            <div class="wallet-option" id="metamaskOption">
                <img src="https://upload.wikimedia.org/wikipedia/commons/3/36/MetaMask_Fox.svg" alt="MetaMask">
                <span>MetaMask</span>
            </div>
            <div class="wallet-option" id="trustWalletOption">
                <img src="https://trustwallet.com/assets/images/media/assets/TWT.png" alt="Trust Wallet">
                <span>Trust Wallet</span>
            </div>
            <div class="wallet-option" id="bitgetWalletOption">
                <img src="https://web.bitget.com/static/image/common/logo.svg" alt="Bitget Wallet">
                <span>Bitget Wallet</span>
            </div>
            <div class="wallet-option" id="walletConnectOption">
                <img src="https://walletconnect.com/_next/static/media/logo_mark.84dd8525.svg" alt="WalletConnect">
                <span>WalletConnect</span>
            </div>
        </div>
    </div>
    
    <!-- Include ethers.js library for blockchain interactions -->
    <script src="ethers.min.js"></script>
    <script>
        // Contract configuration (replace with your actual contract addresses)
        const contractConfig = {
            oldTokenAddress: "0xd7a892f28dEdC74E6b7b33F93BE08abfC394a360", // Replace with your old token address on Arbitrum
            newTokenAddress: "0xe8309a1d00aE006f671A8743a0835bDe092Ac023", // Replace with your new token address on Arbitrum
            swapContractAddress: "0x54246dFb868a3fb3EeF065A563951aB1eE494a8A", // Replace with your swap contract address
            arbitrumChainId: "0xa4b1", // Arbitrum One chain ID (42161 in decimal)
            arbitrumRpcUrl: "https://arb1.arbitrum.io/rpc" // Arbitrum RPC endpoint
        };
        
        // DOM Elements
        const connectWalletBtn = document.getElementById('connectWalletBtn');
        const approveBtn = document.getElementById('approveBtn');
        const swapBtn = document.getElementById('swapBtn');
        const walletOptions = document.getElementById('walletOptions');
        const closeWalletOptions = document.getElementById('closeWalletOptions');
        const oldTokenBalance = document.getElementById('oldTokenBalance');
        const statusMessage = document.getElementById('statusMessage');
        
        // Global variables
        let provider;
        let signer;
        let userAddress;
        let walletName;
        let oldTokenContract;
        let newTokenContract;
        let swapContract;
        
        // Initialize when page loads
        window.addEventListener('load', () => {
            // Check if wallet is already connected
            if (window.ethereum && window.ethereum.selectedAddress) {
                initWalletConnection(window.ethereum, window.ethereum.isMetaMask ? 'MetaMask' : 'Wallet');
            }
            
            // Set up wallet option click handlers
            document.getElementById('metamaskOption').addEventListener('click', () => connectMetaMask());
            document.getElementById('trustWalletOption').addEventListener('click', () => connectTrustWallet());
            document.getElementById('bitgetWalletOption').addEventListener('click', () => connectBitgetWallet());
            document.getElementById('walletConnectOption').addEventListener('click', () => connectWalletConnect());
            
            // Set up other event listeners
            connectWalletBtn.addEventListener('click', () => walletOptions.style.display = 'flex');
            closeWalletOptions.addEventListener('click', () => walletOptions.style.display = 'none');
            
            // Token amount input handler
            document.getElementById('oldTokenAmount').addEventListener('input', (e) => {
                const amount = parseFloat(e.target.value) || 0;
                document.getElementById('newTokenAmount').value = amount.toFixed(6); // 1:1 swap ratio
            });
            
            // Approve button handler
            approveBtn.addEventListener('click', approveTokens);
            
            // Swap button handler
            swapBtn.addEventListener('click', swapTokens);
        });
        
        // Initialize wallet connection
        async function initWalletConnection(ethereumProvider, name) {
            try {
                walletName = name;
                
                // Check if we're on the correct network (Arbitrum)
                await checkNetwork();
                
                // Set up ethers.js provider and signer
                provider = new ethers.providers.Web3Provider(ethereumProvider);
                signer = provider.getSigner();
                userAddress = await signer.getAddress();
                
                // Initialize token contracts
                initializeContracts();
                
                // Update UI
                updateWalletUI();
                loadTokenBalance();
                
                // Set up event listeners for account/chain changes
                ethereumProvider.on('accountsChanged', handleAccountsChanged);
                ethereumProvider.on('chainChanged', handleChainChanged);
                
            } catch (error) {
                showError(`Connection error: ${error.message}`);
                console.error("Wallet connection error:", error);
            }
        }
        
        // Initialize token and swap contracts
        function initializeContracts() {
            // ERC20 ABI (simplified for transfer functions)
            const erc20Abi = [
                "function balanceOf(address owner) view returns (uint256)",
                "function transfer(address to, uint amount) returns (bool)",
                "function approve(address spender, uint amount) returns (bool)",
                "function allowance(address owner, address spender) view returns (uint256)",
                "function transferFrom(address from, address to, uint amount) returns (bool)"
            ];
            
            // Swap contract ABI (from your provided contract)
            const swapContractAbi = [
                "function swapTokens(uint256 amount) external",
                "function swappedAmounts(address) view returns (uint256)",
                "function swapActive() view returns (bool)"
            ];
            
            // Create contract instances
            oldTokenContract = new ethers.Contract(contractConfig.oldTokenAddress, erc20Abi, signer);
            newTokenContract = new ethers.Contract(contractConfig.newTokenAddress, erc20Abi, signer);
            swapContract = new ethers.Contract(contractConfig.swapContractAddress, swapContractAbi, signer);
        }
        
        // Check if we're on Arbitrum network, switch if not
        async function checkNetwork() {
            const chainId = await window.ethereum.request({ method: 'eth_chainId' });
            
            if (chainId !== contractConfig.arbitrumChainId) {
                try {
                    await window.ethereum.request({
                        method: 'wallet_switchEthereumChain',
                        params: [{ chainId: contractConfig.arbitrumChainId }]
                    });
                } catch (switchError) {
                    // This error code indicates that the chain has not been added to MetaMask
                    if (switchError.code === 4902) {
                        try {
                            await window.ethereum.request({
                                method: 'wallet_addEthereumChain',
                                params: [{
                                    chainId: contractConfig.arbitrumChainId,
                                    chainName: 'Arbitrum One',
                                    nativeCurrency: {
                                        name: 'Ethereum',
                                        symbol: 'ETH',
                                        decimals: 18
                                    },
                                    rpcUrls: [contractConfig.arbitrumRpcUrl],
                                    blockExplorerUrls: ['https://arbiscan.io/']
                                }]
                            });
                        } catch (addError) {
                            throw new Error('Failed to add Arbitrum network to wallet');
                        }
                    } else {
                        throw new Error('Failed to switch to Arbitrum network');
                    }
                }
            }
        }
        
        // Load user's old token balance
        async function loadTokenBalance() {
            try {
                const balance = await oldTokenContract.balanceOf(userAddress);
                const decimals = 18; // Assuming standard 18 decimal tokens
                const formattedBalance = ethers.utils.formatUnits(balance, decimals);
                
                oldTokenBalance.textContent = `Balance: ${parseFloat(formattedBalance).toFixed(2)} OLD`;
                
                // Check allowance
                checkAllowance();
                
            } catch (error) {
                console.error("Error loading token balance:", error);
                showError("Failed to load token balance");
            }
        }
        
        // Check token allowance for swap contract
        async function checkAllowance() {
            try {
                const allowance = await oldTokenContract.allowance(userAddress, contractConfig.swapContractAddress);
                const amount = document.getElementById('oldTokenAmount').value;
                const decimals = 18;
                
                if (amount && parseFloat(amount) > 0) {
                    const requiredAllowance = ethers.utils.parseUnits(amount, decimals);
                    
                    if (allowance.gte(requiredAllowance)) {
                        approveBtn.style.display = 'none';
                        swapBtn.style.display = 'block';
                    } else {
                        approveBtn.style.display = 'block';
                        swapBtn.style.display = 'none';
                    }
                }
            } catch (error) {
                console.error("Error checking allowance:", error);
            }
        }
        
        // Approve tokens for swapping
        async function approveTokens() {
            try {
                const amount = document.getElementById('oldTokenAmount').value;
                if (!amount || parseFloat(amount) <= 0) {
                    showError("Please enter a valid amount to approve");
                    return;
                }
                
                const decimals = 18;
                const parsedAmount = ethers.utils.parseUnits(amount, decimals);
                
                showLoading("Approving OLD tokens...");
                
                const tx = await oldTokenContract.approve(contractConfig.swapContractAddress, parsedAmount);
                await tx.wait();
                
                showSuccess("Approval successful!");
                checkAllowance();
                
            } catch (error) {
                console.error("Approval error:", error);
                showError(`Approval failed: ${error.message}`);
            }
        }
        
        // Execute token swap
        async function swapTokens() {
            try {
                const amount = document.getElementById('oldTokenAmount').value;
                if (!amount || parseFloat(amount) <= 0) {
                    showError("Please enter a valid amount to swap");
                    return;
                }
                
                const decimals = 18;
                const parsedAmount = ethers.utils.parseUnits(amount, decimals);
                
                // Check if swap is active
                const isSwapActive = await swapContract.swapActive();
                if (!isSwapActive) {
                    showError("Token swap is not currently active");
                    return;
                }
                
                showLoading("Swapping tokens...");
                
                // Execute swap
                const tx = await swapContract.swapTokens(parsedAmount);
                const receipt = await tx.wait();
                
                showSuccess(`Swap successful! Transaction hash: ${receipt.transactionHash}`);
                
                // Refresh balance and allowance
                loadTokenBalance();
                
            } catch (error) {
                console.error("Swap error:", error);
                showError(`Swap failed: ${error.message}`);
            }
        }
        
        // Wallet connection functions
        async function connectMetaMask() {
            walletOptions.style.display = 'none';
            
            if (window.ethereum && window.ethereum.isMetaMask) {
                try {
                    const accounts = await window.ethereum.request({ method: 'eth_requestAccounts' });
                    await initWalletConnection(window.ethereum, 'MetaMask');
                } catch (error) {
                    showError(`MetaMask connection error: ${error.message}`);
                }
            } else {
                showError('MetaMask is not installed. Please install it from https://metamask.io/');
            }
        }
        
        async function connectTrustWallet() {
            walletOptions.style.display = 'none';
            
            if (window.ethereum && window.ethereum.isTrust) {
                try {
                    const accounts = await window.ethereum.request({ method: 'eth_requestAccounts' });
                    await initWalletConnection(window.ethereum, 'Trust Wallet');
                } catch (error) {
                    showError(`Trust Wallet connection error: ${error.message}`);
                }
            } else {
                showError('Trust Wallet is not installed or not detected.');
            }
        }
        
        async function connectBitgetWallet() {
            walletOptions.style.display = 'none';
            
            if (window.bitkeep && window.bitkeep.ethereum) {
                try {
                    const accounts = await window.bitkeep.ethereum.request({ method: 'eth_requestAccounts' });
                    await initWalletConnection(window.bitkeep.ethereum, 'Bitget Wallet');
                } catch (error) {
                    showError(`Bitget Wallet connection error: ${error.message}`);
                }
            } else {
                showError('Bitget Wallet is not installed or not detected.');
            }
        }
        
        async function connectWalletConnect() {
            walletOptions.style.display = 'none';
            showError("WalletConnect integration would be implemented here");
            // In a real implementation, you would use the WalletConnect library
        }
        
        // Update UI after wallet connection
        function updateWalletUI() {
            const shortAddress = `${userAddress.substring(0, 6)}...${userAddress.substring(userAddress.length - 4)}`;
            connectWalletBtn.textContent = `${walletName}: ${shortAddress}`;
            connectWalletBtn.style.backgroundColor = '#666';
        }
        
        // Handle account changes
        function handleAccountsChanged(accounts) {
            if (accounts.length === 0) {
                // Wallet disconnected
                resetWalletConnection();
            } else if (accounts[0] !== userAddress) {
                // Account changed
                userAddress = accounts[0];
                updateWalletUI();
                loadTokenBalance();
            }
        }
        
        // Handle chain changes
        function handleChainChanged(chainId) {
            if (chainId !== contractConfig.arbitrumChainId) {
                showError("Please switch to Arbitrum network");
                resetWalletConnection();
            } else {
                window.location.reload();
            }
        }
        
        // Reset wallet connection
        function resetWalletConnection() {
            connectWalletBtn.textContent = 'Connect Wallet';
            connectWalletBtn.style.backgroundColor = '#4CAF50';
            approveBtn.style.display = 'none';
            swapBtn.style.display = 'none';
            oldTokenBalance.textContent = 'Balance: 0';
            document.getElementById('oldTokenAmount').value = '';
            document.getElementById('newTokenAmount').value = '';
        }
        
        // UI helper functions
        function showLoading(message) {
            statusMessage.innerHTML = `<span class="loading"></span>${message}`;
            statusMessage.className = 'status-message';
            statusMessage.style.display = 'block';
        }
        
        function showSuccess(message) {
            statusMessage.innerHTML = message;
            statusMessage.className = 'status-message success';
            statusMessage.style.display = 'block';
        }
        
        function showError(message) {
            statusMessage.innerHTML = message;
            statusMessage.className = 'status-message error';
            statusMessage.style.display = 'block';
        }
    </script>
</body>
</html>